// Array
Array.contains = function (array, value) {
    if (!Array.isArray(array))
        nav_throwError("The argument for Array.contains should be array");

    for (var i = 0; i < array.length; i++) {
        if (array[i] == value)
            return true;
    }
    return false;
};

Array.unique = function (array) {
    if (!Array.isArray(array))
        nav_throwError("The argument for Array.unique should be array");

    var uniqueArray = [];
    for (var i = 0; i < array.length; i++) {
        var obj = array[i];
        if (!Array.contains(uniqueArray, obj)) {
            uniqueArray.push(obj);
        }
    }
    return uniqueArray;
};

Array.shuffle = function (array) {
    if (!Array.isArray(array))
        nav_throwError("The argument for Array.shuffle should be array");

    for (var i = 0; i < array.length - 1; i++) {
        var newPos = i + Math.floor(Math.random() * (array.length - i));
        var oriItem = array[i];
        array[i] = array[newPos];
        array[newPos] = oriItem;
    }
    return array;
};

Array.tojson = function (array, indent, nolint, depth) {
    if (!Array.isArray(array))
        nav_throwError("The argument for Array.tojson should be array");
    
    if (typeof depth !== 'number') {
        depth = 0;
    }
    if (depth > tojson.MAX_DEPTH) {
        return "[Array]";
    }

    var elementSeparator = nolint ? " " : "\n";

    if (!indent)
        indent = "";
    if (nolint)
        indent = "";

    if (array.length == 0) {
        return "[ ]";
    }

    var s = "[" + elementSeparator;

    if (!nolint)
        indent += "\t";

    for (var i = 0; i < array.length; i++) {
        s += indent + tojson(array[i], indent, nolint, depth + 1);
        if (i < array.length - 1) {
            s += "," + elementSeparator;
        }
    }

    if (!nolint)
        indent = indent.substring(1);

    s += elementSeparator + indent + "]";
    return s;
};

Array.fetchRefs = function (arr, coll) {
    if (!Array.isArray(arr)) {
        throw new Error("The first argument to Array.fetchRefs must be an array");
    }

    var n = [];
    for (var i = 0; i < arr.length; i++) {
        var z = arr[i];
        if (coll && coll != z.getCollection())
            continue;
        n.push(z.fetch());
    }
    return n;
};

Array.sum = function (array) {
    if (!Array.isArray(array))
        nav_throwError("The argument for Array.sum should be array");

    if (array.length == 0)
        return null;
    var sum = array[0];
    for (var i = 1; i < array.length; i++)
        sum += array[i];
    return sum;
};

Array.avg = function (array) {
    if (!Array.isArray(array))
        nav_throwError("The argument for Array.avg should be array");

    if (array.length == 0)
        return null;
    return Array.sum(array) / array.length;
};

Array.stdDev = function (array) {
    if (!Array.isArray(array))
        nav_throwError("The argument for Array.stdDev should be array");
    
    var average = Array.avg(array);
    var sum = 0;
    for (var i = 0; i < array.length; i++)
        sum += Math.pow(array[i] - average, 2);
    return Math.sqrt(sum / array.length);
};

Array.equals = function (array1, array2) {
    if (!array1 || !array2)
        return false;

    if (array1.length != array2.length)
        return false;

    for (var i = 0, l = array1.length; i < l; i++) {
        if (array1[i] instanceof Array && array2[i] instanceof Array) {
            if (!array1[i].equals(array2[i]))
                return false;
        } else if (array1[i] != array2[i]) {
            return false;
        }
    }
    return true;
}

// BinData
BinData.prototype.toString = function () {
    return this.forwardToCustomFunction("binDataToString");
};
BinData.prototype.tojson = BinData.prototype.toString;

BinData.prototype.toJSON = function () {
    return this.forwardToCustomFunction("binDataToJSON");
};

BinData.prototype.subtype = function () {
    return this.forwardToCustomFunction("binDataSubtype");
};
BinData.prototype.length = function () {
    return this.forwardToCustomFunction("binDataLength");
};
BinData.prototype.base64 = function () {
    return this.forwardToCustomFunction("binDataToBase64");
};
BinData.prototype.hex = function () {
    return this.forwardToCustomFunction("binDataToHex");
};

// Date
Date.timeFunc = function (jsFunction, numberOfTimes) {
    var start = new Date();
    numberOfTimes = numberOfTimes || 1;
    for (var i = 0; i < numberOfTimes; i++) {
        jsFunction.apply(null, Array.from(arguments).slice(2));
    }

    return (new Date()).getTime() - start.getTime();
};

Date.prototype.tojson = function () {
    var UTC = 'UTC';
    var year = this['get' + UTC + 'FullYear']().zeroPad(4);
    var month = (this['get' + UTC + 'Month']() + 1).zeroPad(2);
    var date = this['get' + UTC + 'Date']().zeroPad(2);
    var hour = this['get' + UTC + 'Hours']().zeroPad(2);
    var minute = this['get' + UTC + 'Minutes']().zeroPad(2);
    var second = this['get' + UTC + 'Seconds']().zeroPad(2);

    if (this['get' + UTC + 'Milliseconds']())
        second += '.' + this['get' + UTC + 'Milliseconds']().zeroPad(3);

    var offset = 'Z';
    return 'ISODate("' + year + '-' + month + '-' + date + 'T' + hour + ':' + minute + ':' + second +
        offset + '")';
};

ISODate = function (isoDateStr) {
    if (!isoDateStr)
        return new Date();

    var isoDateRegex =
        /^(\d{4})-?(\d{2})-?(\d{2})([T ](\d{2})(:?(\d{2})(:?(\d{2}(\.\d+)?))?)?(Z|([+-])(\d{2}):?(\d{2})?)?)?$/;
    var res = isoDateRegex.exec(isoDateStr);

    if (!res)
        nav_throwError("invalid ISO date: " + isoDateStr);

    var year = parseInt(res[1], 10);
    var month = (parseInt(res[2], 10)) - 1;
    var date = parseInt(res[3], 10);
    var hour = parseInt(res[5], 10) || 0;
    var minute = parseInt(res[7], 10) || 0;
    var second = parseInt((res[9] && res[9].substr(0, 2)), 10) || 0;
    var millisecond = Math.round((parseFloat(res[10]) || 0) * 1000);

    var dateTime = new Date();

    dateTime.setUTCFullYear(year, month, date);
    dateTime.setUTCHours(hour);
    dateTime.setUTCMinutes(minute);
    dateTime.setUTCSeconds(second);
    var time = dateTime.setUTCMilliseconds(millisecond);

    if (res[11] && res[11] != 'Z') {
        var offset = 0;
        offset += (parseInt(res[13], 10) || 0) * 60 * 60 * 1000;
        offset += (parseInt(res[14], 10) || 0) * 60 * 1000;
        if (res[12] == '+')
            offset *= -1;

        time += offset;
    }

    const DATE_RANGE_MIN_MICROSECONDS = -62167219200000;
    const DATE_RANGE_MAX_MICROSECONDS = 253402300799999;

    if (time < DATE_RANGE_MIN_MICROSECONDS || time > DATE_RANGE_MAX_MICROSECONDS)
        nav_throwError("invalid ISO date: " + isoDateStr);

    return new Date(time);
};

// Code
Code.prototype.toString = function () {
    return this.forwardToCustomFunction("codeToString");
};

// DBPointer
DBPointer.prototype.getCollection = function () {
    return this.ns;
};

DBPointer.prototype.getId = function () {
    return this.id;
};

DBPointer.prototype.toString = function () {
    return this.forwardToCustomFunction("dbPointerToString");
};
DBPointer.prototype.tojson = DBPointer.prototype.toString;

// DBRef
DBRef.prototype.getRef = function () {
    return this.$ref;
};

DBRef.prototype.getCollection = DBRef.prototype.getRef;

DBRef.prototype.getId = function () {
    return this.$id;
};

DBRef.prototype.getDb = function () {
    return this.$db;
};

DBRef.prototype.toString = function () {
    return this.forwardToCustomFunction("dbRefToString");
};
DBRef.prototype.tojson = DBRef.prototype.toString;

MaxKey.tojson = function () {
    return this.forwardToCustomFunction("maxKeyToJson");
};

MaxKey.toJSON = function () {
    return this.forwardToCustomFunction("maxKeyToJSON");
};

MinKey.tojson = function () {
    return this.forwardToCustomFunction("minKeyToJson");
};

MinKey.toJSON = function () {
    return this.forwardToCustomFunction("minKeyToJSON");
};

// Number
Number.prototype.toPercentStr = function () {
    return (this * 100).toFixed(2) + "%";
};

Number.prototype.zeroPad = function (width) {
    return ('' + this).pad(width, false, '0');
};

// NumberDecimal
NumberDecimal.prototype.toString = function () {
    return this.forwardToCustomFunction("numberDecimalToString");
};
NumberDecimal.prototype.tojson = NumberDecimal.prototype.toString;

NumberDecimal.prototype.toJSON = function () {
    return this.forwardToCustomFunction("numberDecimalToJSON");
};

// Decimal128
Decimal128 = function (value) {
    return NumberDecimal(value);
};

// NumberInt
NumberInt.prototype.toString = function () {
    return this.forwardToCustomFunction("numberIntToString");
};
NumberInt.prototype.tojson = NumberInt.prototype.toString;

NumberInt.prototype.valueOf = function () {
    return this.forwardToCustomFunction("numberIntToValueOf");
};

NumberInt.prototype.toJSON = function () {
    return this.forwardToCustomFunction("numberIntToJSON");
};

// Int32
Int32 = function (value) {
    return NumberInt(value);
};

// NumberLong
NumberLong.prototype.toString = function () {
    return this.forwardToCustomFunction("numberLongToString");
};
NumberLong.prototype.tojson = NumberLong.prototype.toString;

NumberLong.prototype.valueOf = function () {
    return this.forwardToCustomFunction("numberLongToValueOf");
};

NumberLong.prototype.toJSON = function () {
    return this.forwardToCustomFunction("numberLongToJSON");
};

// Long
Long = function (value) {
    return NumberLong(value);
};

// ObjectId
ObjectId.prototype.toString = function () {
    return this.forwardToCustomFunction("objectIdToString");
};
ObjectId.prototype.tojson = ObjectId.prototype.toString;

ObjectId.prototype.toJSON = function () {
    return this.forwardToCustomFunction("objectIdToJSON");
};

ObjectId.prototype.valueOf = function () {
    return this.str;
};

ObjectId.prototype.isObjectId = true;

ObjectId.prototype.getTimestamp = function () {
    return new Date(parseInt(this.valueOf().slice(0, 8), 16) * 1000);
};

ObjectId.prototype.equals = function (otherObjectId) {
    return this.str == otherObjectId.str;
};

ObjectId.fromDate = function (source) {
    if (!source)
        nav_throwError("date missing or undefined");

    var sourceDate;

    if (source instanceof Date)
        sourceDate = source;
    else
        nav_throwError("Cannot create ObjectId from " + typeof (source) + ": " + tojson(source));

    var seconds = Math.floor(sourceDate.getTime() / 1000);

    var hexTimestamp = seconds.toString(16).pad(8, false, '0') + "0000000000000000";

    var objectId = ObjectId(hexTimestamp);

    return objectId;
};

// Regular Expression
RegExp.escape = function (text) {
    return text.replace(/[-[\]{}()*+?.,\\^$|#\s]/g, "\\$&");
};

RegExp.prototype.tojson = RegExp.prototype.toString;

// String
if (isUndefined(String.prototype.trim)) {
    String.prototype.trim = function () {
        return this.replace(/^\s+|\s+$/g, "");
    };
}
if (isUndefined(String.prototype.trimLeft)) {
    String.prototype.trimLeft = function () {
        return this.replace(/^\s+/, "");
    };
}
if (isUndefined(String.prototype.trimRight)) {
    String.prototype.trimRight = function () {
        return this.replace(/\s+$/, "");
    };
}

String.prototype.ltrim = String.prototype.trimLeft;
String.prototype.rtrim = String.prototype.trimRight;

String.prototype.startsWith = function (str) {
    return this.indexOf(str) == 0;
};

String.prototype.endsWith = function (str) {
    return this.indexOf(str, this.length - str.length) !== -1;
};

if (!String.prototype.includes) {
    String.prototype.includes = function () {
        'use strict';
        return String.prototype.indexOf.apply(this, arguments) !== -1;
    };
}

String.prototype.pad = function (length, right, char) {
    if (isUndefined(char))
        char = ' ';
    var resultStr = this;
    for (var i = length - resultStr.length; i > 0; i--) {
        if (right)
            resultStr = resultStr + char;
        else
            resultStr = char + resultStr;
    }
    return resultStr;
};

// Timestamp
Timestamp.prototype.getTime = function () {
    return this.t;
};

Timestamp.prototype.getInc = function () {
    return this.i;
};

Timestamp.prototype.toString = function () {
    return this.forwardToCustomFunction("timestampToString");
};
Timestamp.prototype.tojson = Timestamp.prototype.toString;

Timestamp.prototype.toJSON = function () {
    return this.forwardToCustomFunction("timestampToJSON");
};

// UUID
function _uuid() {
    var d = Date.now();
    return 'xxxxxxxx-xxxx-4xxx-yxxx-xxxxxxxxxxxx'.replace(/[xy]/g, function (c) {
        var r = (d + Math.random() * 16) % 16 | 0;
        d = Math.floor(d / 16);
        return (c === 'x' ? r : (r & 0x3 | 0x8)).toString(16);
    });
}

UUID = function newUUID(uuid) {
    uuid = uuid || _uuid();
    if (!isString(uuid))
        nav_throwError("Invalid UUID string: " + tojson(uuid));

    var hex = uuid.replace(/[{}-]/g, "");
    if (isString(hex) && hex.length != 32)
        nav_throwError("Invalid UUID string: " + hex);
        
    var base64 = HexToBase64(hex);
    return new BinData(4, base64);
}
